package gg.now.nowggsdkdemo.login;

import android.accounts.Account;
import android.accounts.AccountManager;
import android.accounts.AccountManagerCallback;
import android.accounts.AccountManagerFuture;
import android.accounts.AuthenticatorException;
import android.accounts.OperationCanceledException;
import android.content.ActivityNotFoundException;
import android.content.ComponentName;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.os.Parcel;
import android.os.ResultReceiver;
import android.util.Log;
import android.view.View;
import android.widget.TextView;

import androidx.activity.EdgeToEdge;
import androidx.appcompat.app.AppCompatActivity;
import androidx.core.graphics.Insets;
import androidx.core.view.ViewCompat;
import androidx.core.view.WindowInsetsCompat;

import com.google.gson.Gson;
import com.google.gson.annotations.SerializedName;

import java.io.IOException;

import gg.now.nowggsdkdemo.R;
import okhttp3.OkHttpClient;
import retrofit2.Call;
import retrofit2.Callback;
import retrofit2.Response;
import retrofit2.Retrofit;
import retrofit2.converter.gson.GsonConverterFactory;
import retrofit2.converter.scalars.ScalarsConverterFactory;
import retrofit2.http.Body;
import retrofit2.http.POST;

public class LoginActivity extends AppCompatActivity {

    private final int GET_ACCOUNTS_REQUEST_CODE = 100;
    private static final String TAG = "MainActivity";
    public static final String ID_TOKEN = "id_token"; // Do Not Modify
    public static final String CLIENT_ID = "zBC1LCs7s7IuZzxQP9oO$$01FVCMAVCTZWJKCRPY4EH339SG"; // Replace with your client id
    public static final String ACCOUNT_TYPE = "now.gg"; // Do Not Modify
    public static final String HOST_URL = "hostUrl"; //Do Not Modify

    @Override
    public void onRequestPermissionsResult(int requestCode,
                                           String permissions[], int[] grantResults) {
        super.onRequestPermissionsResult(requestCode, permissions, grantResults);
        if (requestCode == GET_ACCOUNTS_REQUEST_CODE) {
            if (grantResults.length > 0 && grantResults[0] == PackageManager.PERMISSION_GRANTED) {
                // Permission granted, can proceed with sign-in.
                signIn();
            }
        }
    }

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        EdgeToEdge.enable(this);
        setContentView(R.layout.activity_main_login);
        ViewCompat.setOnApplyWindowInsetsListener(findViewById(R.id.main), (v, insets) -> {
            Insets systemBars = insets.getInsets(WindowInsetsCompat.Type.systemBars());
            v.setPadding(systemBars.left, systemBars.top, systemBars.right, systemBars.bottom);
            return insets;
        });
        if (checkSelfPermission(android.Manifest.permission.GET_ACCOUNTS) != PackageManager.PERMISSION_GRANTED) {
            requestPermissions(new String[] {
                            android.Manifest.permission.GET_ACCOUNTS
                    },
                    GET_ACCOUNTS_REQUEST_CODE);
        } else {
            // has GET_ACCOUNTS permission, can proceed with signin
            Log.d(TAG, "onCreate: Already has permission, proceed with signin");
        }
        findViewById(R.id.now_gg_login_button).setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                signIn();
            }
        });
    }

    private void signIn() {
        Account account = getNowggAccount();
        if (account != null) {
            Bundle bundle = new Bundle();
            bundle.putString("client_id", CLIENT_ID);
            String authTokenType = ID_TOKEN;
            AccountManager.get(getApplicationContext()).
                    getAuthToken(account, authTokenType, bundle, LoginActivity.this, new OnTokenAcquired(), null);
        }
        else {
            addNowggAccount();
        }
    }

    private Account getNowggAccount() {
        Account[] accounts = AccountManager.get(getApplicationContext()).getAccountsByType(ACCOUNT_TYPE);
        if (accounts.length > 0) {
            Log.d(TAG, "getNowggAccount: account found");
            // currently only one now.gg account can be added in a system
            return accounts[0];
        }
        return null;
    }

    ResultReceiver parcelResultReceiver(ResultReceiver actualReceiver) {
        Parcel parcel = Parcel.obtain();
        actualReceiver.writeToParcel(parcel,0);
        parcel.setDataPosition(0);
        ResultReceiver receiverForSending = ResultReceiver.CREATOR.createFromParcel(parcel);
        parcel.recycle();
        return receiverForSending;
    }

    ResultReceiver resultReceiver = new ResultReceiver(new Handler(Looper.getMainLooper())) {
        @Override
        protected void onReceiveResult(int resultCode, Bundle resultData) {
            super.onReceiveResult(resultCode, resultData);
            Log.d(TAG, "onReceiveResult() called with: resultCode = [" + resultCode + "], resultData = [" + resultData.toString() + "]");
            // onReceiveResult
            if (resultCode == 0) { // means sign in is okay
                signIn();
            } else {
                // sign in failed
                Log.d(TAG, "onReceiveResult: sign in failed");
            }
        }
    };

    private void addNowggAccount() {
        try {
            Intent intent = new Intent();
            intent.setComponent(new ComponentName("gg.now.accounts", "gg.now.accounts.AuthenticatorActivity"));
            intent.setAction("IAP_ADD_ACCOUNT");
            intent.putExtra("resultReceiver", parcelResultReceiver(resultReceiver));
            intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
            startActivity(intent);
        } catch (ActivityNotFoundException e) {
            e.printStackTrace();
        }
    }

    private class OnTokenAcquired implements AccountManagerCallback {

        @Override
        public void run(AccountManagerFuture result) {
            try {
                Bundle bundle = (Bundle) result.getResult();
                boolean success = bundle.getBoolean(AccountManager.KEY_BOOLEAN_RESULT, false);
                if (success) {
                    final String token = bundle.getString(ID_TOKEN);
                    // get the now.gg server URL, from which you can get the user details and verify the tokens

                    final String hostUrl = bundle.getString(HOST_URL);
                    verifyToken(token, hostUrl);
                }
                else {
                    // get token failed
                    // error case. The developer can show errors or show other login mechanisms
                    Log.d(TAG, "run: get token failed " + bundle);
                }
            } catch (AuthenticatorException | IOException | OperationCanceledException e) {
                e.printStackTrace();
            }
        }
    }

    private void verifyToken(String idToken, String hostUrl) {
        BackendApiCallService backendApiCallService = getBackendRetrofit(hostUrl).create(BackendApiCallService.class);

        Call<TokenVerifyResponse> tokenVerifyResponseCall = backendApiCallService.verifyIdToken(new TokenVerifyRequest(ID_TOKEN, idToken, CLIENT_ID));

        tokenVerifyResponseCall.enqueue(new Callback<TokenVerifyResponse>() {
            @Override
            public void onResponse(Call<TokenVerifyResponse> call, Response<TokenVerifyResponse> response) {
                if (response.isSuccessful()) {
                    Log.d(TAG, "onResponse: " + response.body().toString());
                    ((TextView)findViewById(R.id.textView)).setText(response.body().toString());
                }
                else {
                    Gson gson = new Gson();
                    TokenVerifyResponse error = gson.fromJson(response.errorBody().charStream(), TokenVerifyResponse.class);

                    if (!error.isSuccess() && ("EXPIRED_TOKEN".equals(error.getCode()) || "INVALID_TOKEN".equals(error.getCode()))) {
                        // retry the verify token call, after getting a new TOKEN
                    }
                }
            }
            @Override
            public void onFailure(Call call, Throwable t) {
                Log.d(TAG, "onFailure() called with: call = [" + call + "], t = [" + t + "]");
            }
        });

    }

    public Retrofit getBackendRetrofit(String hostUrl) {
        OkHttpClient okHttpClient = new OkHttpClient.Builder().build();
        return new Retrofit.Builder()
                .baseUrl(hostUrl)
                .addConverterFactory(ScalarsConverterFactory.create())
                .addConverterFactory(GsonConverterFactory.create())
                .client(okHttpClient)
                .build();
    }

    public interface BackendApiCallService {
        @POST("/accounts/oauth2/v1/verify-token")
        Call<TokenVerifyResponse> verifyIdToken(@Body TokenVerifyRequest authRequest);
    }
    public class TokenVerifyRequest {
        String token_type;
        String token;
        String client_id;

        public TokenVerifyRequest(String token_type, String token, String client_id) {
            this.token_type = token_type;
            this.token = token;
            this.client_id = client_id;
        }
    }

    public class TokenVerifyResponse {
        boolean success;
        String code;
        String msg;
        @SerializedName("decodedData")
        UserDataVerified userDataVerified;

        public TokenVerifyResponse(boolean success, String code, String msg, UserDataVerified userDataVerified) {
            this.success = success;
            this.code = code;
            this.msg = msg;
            this.userDataVerified = userDataVerified;
        }

        public UserDataVerified getUserDataVerified() {
            return userDataVerified;
        }

        public boolean isSuccess() {
            return success;
        }
        public String getCode() {
            return code;
        }

        @Override
        public String toString() {
            return "TokenVerifyResponse{" +
                    "success=" + success +
                    ", code='" + code + '\'' +
                    ", msg='" + msg + '\'' +
                    ", userDataVerified=" + userDataVerified +
                    '}';
        }
    }

    public class UserDataVerified {
        String iss;
        String sub;
        String aud;
        String exp;
        String iat;
        String auth_time;
        String tokenId;
        String sessionId;
        String scope;

        String email;
        String name;
        String picture;
        String mobile;
        String userId;

        public UserDataVerified(String iss, String sub, String aud, String exp, String iat,
                                String auth_time, String email, String mobile, String userId,
                                String tokenId, String sessionId, String scope, String name, String picture) {
            this.iss = iss;
            this.sub = sub;
            this.aud = aud;
            this.exp = exp;
            this.iat = iat;
            this.auth_time = auth_time;
            this.email = email;
            this.mobile = mobile;
            this.userId = userId;
            this.tokenId = tokenId;
            this.sessionId = sessionId;
            this.scope = scope;
            this.name = name;
            this.picture = picture;
        }

        @Override
        public String toString() {
            return "UserDataVerified{" +
                    "iss='" + iss + '\'' +
                    ", sub='" + sub + '\'' +
                    ", aud='" + aud + '\'' +
                    ", exp='" + exp + '\'' +
                    ", iat='" + iat + '\'' +
                    ", auth_time='" + auth_time + '\'' +
                    ", tokenId='" + tokenId + '\'' +
                    ", sessionId='" + sessionId + '\'' +
                    ", scope='" + scope + '\'' +
                    ", email='" + email + '\'' +
                    ", name='" + name + '\'' +
                    ", picture='" + picture + '\'' +
                    ", mobile='" + mobile + '\'' +
                    ", userId='" + userId + '\'' +
                    '}';
        }
    }
}