package gg.now.nowggsdkdemo.payments;

import android.content.Context;
import android.content.SharedPreferences;
import android.util.Log;

import java.util.List;

import gg.now.billingclient.api.BillingClient.BillingResponse;
import gg.now.billingclient.api.Purchase;
import gg.now.nowggsdkdemo.R;
import gg.now.nowggsdkdemo.payments.billing.BillingManager;
import gg.now.nowggsdkdemo.payments.skulist.row.DiamondsDelegate;
import gg.now.nowggsdkdemo.payments.skulist.row.StarterPackageDelegate;
import gg.now.nowggsdkdemo.payments.skulist.row.SwordDelegate;
import gg.now.nowggsdkdemo.payments.skulist.row.VIPMonthlyDelegate;

/**
 * Handles control logic of the BillingActivity
 */
class MainViewController {
    private static final String TAG = "MainViewController";

    private final UpdateListener mUpdateListener;
    private final BillingActivity mActivity;

    private SharedPreferences sharedPreferences;

    // Tracks if we currently own subscriptions/inapp SKUs
    private boolean mVIPMonthly;

    private int quantityOfSwords;
    private int quantityOfPackages;
    private int quantityOfDiamonds;

    private String purchaseTokenSword = "";
    private String purchaseTokenPackage = "";
    private String purchaseTokenDiamonds = "";

    private String purchaseTokenVIPMonthly = "";

    MainViewController(BillingActivity activity) {
        mUpdateListener = new UpdateListener();
        mActivity = activity;
        sharedPreferences = activity.getSharedPreferences("payments", Context.MODE_PRIVATE);
        quantityOfSwords = sharedPreferences.getInt("quantityOfSwords", 0);
        quantityOfPackages = sharedPreferences.getInt("quantityOfPackages", 0);
        quantityOfDiamonds = sharedPreferences.getInt("quantityOfDiamonds", 0);
    }

    UpdateListener getUpdateListener() {
        return mUpdateListener;
    }

    boolean isVIPMonthlySubscribed() {
        return mVIPMonthly;
    }

    boolean isSwordPurchased() {
        return quantityOfSwords > 0;
    }

    boolean isPackagePurchased() {
        return quantityOfPackages > 0;
    }

    boolean isDiamondsPurchased() {
        return quantityOfDiamonds > 0;
    }

    public int getQuantityOfSwords() {
        return quantityOfSwords;
    }

    public int getQuantityOfPackages() {
        return quantityOfPackages;
    }

    public int getQuantityOfDiamonds() {
        return quantityOfDiamonds;
    }

    /**
     * Handler to billing updates
     */
    private class UpdateListener implements BillingManager.BillingUpdatesListener {
        @Override
        public void onBillingClientSetupFinished() {
            mActivity.onBillingManagerSetupFinished();
        }

        @Override
        public void onConsumeFinished(String token, @BillingResponse int result) {
            Log.i(TAG, "Consumption finished. Purchase token: " + token + ", result: " + result);

            if (result == BillingResponse.OK) {
                Log.i(TAG, "Consumption successful. Provisioning.");
                if (purchaseTokenSword.equals(token)) {
                    quantityOfSwords++;
                } else if (purchaseTokenPackage.equals(token)) {
                    quantityOfPackages++;
                } else if (purchaseTokenDiamonds.equals(token)) {
                    quantityOfDiamonds++;
                }
            } else {
                mActivity.alert(R.string.alert_error_consuming, Utils.getResponseString(result));
            }

            sharedPreferences.edit()
                    .putInt("quantityOfSwords", quantityOfSwords)
                    .putInt("quantityOfPackages", quantityOfPackages)
                    .putInt("quantityOfDiamonds", quantityOfDiamonds)
                    .apply();

            mActivity.showRefreshedUi();
            Log.i(TAG, "End consumption flow.");
        }

        @Override
        public void onAcknowledgeFinished(int result) {
            if (result == BillingResponse.OK) {
                Log.i(TAG, "Acknowledgement successful. Provisioning.");
                mVIPMonthly = true;
                mActivity.getBillingManager().queryPurchasesAsync();
            } else {
                mActivity.alert(R.string.alert_error_acknowledging, Utils.getResponseString(result));;
            }
            Log.i(TAG, "Acknowledgement finished. result: " + result);
            mActivity.showRefreshedUi();
        }

        @Override
        public void onPurchasesUpdated(List<Purchase> purchaseList) {
            for (Purchase purchase : purchaseList) {
                Log.i(TAG, "purchase.getPurchaseToken() " + purchase.getPurchaseToken());
                switch (purchase.getSku()) {
                    case SwordDelegate.SKU_ID:
                        Log.i(TAG, "We have sword. To be consumed!");
                        purchaseTokenSword = purchase.getPurchaseToken();
                        mActivity.getBillingManager().consumeAsync(purchase.getPurchaseToken());
                        break;
                    case StarterPackageDelegate.SKU_ID:
                        Log.i(TAG, "We have starter package. To be consumed!");
                        purchaseTokenPackage = purchase.getPurchaseToken();
                        mActivity.getBillingManager().consumeAsync(purchase.getPurchaseToken());
                        break;
                    case DiamondsDelegate.SKU_ID:
                        Log.i(TAG, "We have 10 diamonds. To be consumed!");
                        purchaseTokenDiamonds = purchase.getPurchaseToken();
                        mActivity.getBillingManager().consumeAsync(purchaseTokenDiamonds);
                        break;
                    case VIPMonthlyDelegate.SKU_ID:
                        if (purchase.isAcknowledged() && purchase.getSubscriptionStatus().equals("ACTIVE") ) {
                            Log.i(TAG, "We have VIP monthly. Already acknowledged!");
                            mVIPMonthly = true;
                        } else {
                            Log.i(TAG, "We have VIP monthly. To be acknowledged!");
                            purchaseTokenVIPMonthly = purchase.getPurchaseToken();
                            if (purchase.getSubscriptionStatus().equals("ACTIVE") && !purchase.isAcknowledged()){
                                mActivity.getBillingManager().acknowledgeAsync(purchaseTokenVIPMonthly);
                            }
                        }
                        break;
                }
            }
            mActivity.showRefreshedUi();
        }
    }
}
