package gg.now.nowggsdkdemo.payments.billing;

import android.app.Activity;
import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.util.Log;
import android.widget.Toast;

import androidx.annotation.NonNull;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import gg.now.billingclient.api.AcknowledgePurchaseParams;
import gg.now.billingclient.api.AcknowledgePurchaseResponseListener;
import gg.now.billingclient.api.BillingClient;
import gg.now.billingclient.api.BillingClient.BillingResponse;
import gg.now.billingclient.api.BillingClient.FeatureType;
import gg.now.billingclient.api.BillingClient.SkuType;
import gg.now.billingclient.api.BillingClientStateListener;
import gg.now.billingclient.api.BillingFlowParams;
import gg.now.billingclient.api.BillingResult;
import gg.now.billingclient.api.Constants;
import gg.now.billingclient.api.ConsumeResponseListener;
import gg.now.billingclient.api.ProductDetails;
import gg.now.billingclient.api.ProductDetailsResponseListener;
import gg.now.billingclient.api.Purchase;
import gg.now.billingclient.api.Purchase.PurchasesResult;
import gg.now.billingclient.api.PurchasesResponseListener;
import gg.now.billingclient.api.PurchasesUpdatedListener;
import gg.now.billingclient.api.QueryProductDetailsParams;
import gg.now.billingclient.api.QueryPurchasesParams;
import gg.now.billingclient.api.SkuDetails;
import gg.now.billingclient.api.SkuDetailsParams;
import gg.now.billingclient.api.SkuDetailsResponseListener;
import gg.now.billingclient.util.BillingHelper;
import gg.now.nowggsdkdemo.payments.Utils;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;

/**
 * Handles all the interactions with billing service (via Billing library), maintains connection to
 * it through BillingClient and caches temporary states/data if needed
 */
public class BillingManager implements PurchasesUpdatedListener {
    // Default value of mBillingClientResponseCode until BillingManager was not yet initialized
    public static final int BILLING_MANAGER_NOT_INITIALIZED = -1;

    private static final String TAG = "BillingManager";

    //TODO Replace with your app's id.
    private static final String APP_ID = "10156";
    private static final String APP_ID_QA = "10156";

    //TODO Replace with your inGameId.

    /*TODO BASE_64_ENCODED_PUBLIC_KEY should be YOUR APPLICATION'S PUBLIC KEY
     * (that you got from the developer console). This is not your
     * developer public key, it's the *app-specific* public key.
     *
     * Instead of just storing the entire literal string here embedded in the
     * program,  construct the key at runtime from pieces or
     * use bit manipulation (for example, XOR with some other string) to hide
     * the actual key.  The key itself is not secret information, but we don't
     * want to make it easy for an attacker to replace the public key with one
     * of their own and then fake messages from the server.
     */
    private static final String BASE_64_ENCODED_PUBLIC_KEY =
            "MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAygWG1GhaNALZ9K6O+a6QKVEuQ9EGvUz2A6Ar2B0zBGe8hHwvdrr507SUpVkxF00NfZQxjljCkOFbOA/hvM+rZGkdZem2uG0O0x2q6d3ddGbwXjKSJrZumxsrtXtH4H8cwHGnVJbK5Jj1tEDijSCU3RF/Bl493PvaA479Suv4JRpaRSCUPRn1OzxXmtJaGyj4kFT51sYfwJ2ar0O4j98Be51wu7qcf/941CaWAyYJ6+heCahOEr4+85/njbLm35R+tz1aRpKIjerSw3fBX1OkfnCaruFPVtf1llsDlt2HE+ExcG0+lNGfDuw1yP2bVNODNl1mPKXloXPlsIOR28e2zwIDAQAB";
    private static final String BASE_64_ENCODED_PUBLIC_KEY_QA =
            "MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEA3CCd8CY0AeerM0KO8NYVcC6g5Dr3N4AWC7O/bP/7SyTOyRr66/9HfiVEKV/3d3fP4Pck29m1jimAcakjf7r3FO0TBC/D58nzBmXZbfV7uhATNjGbrFZoau8HY8FqkCk6b/xbDpSa9/tTxk6JQquX/Txp9sfgdjW/xD7WQaHaiyJGeTgwubkYA50qgloYoLQF+mGr2jxygHbP6pQ8Gso7tokiOuWkJSO9TNCNB5EXc15D1QLH2++YbRcDTTot6xYzUuzwnqDmJiefrlXpHZ67vm8LOeSEjGJnW1FxJdht27AeLijY98CBeYEA/95caktnJlBXvwcVPlD6xdwDslp6PwIDAQAB";

    private final BillingUpdatesListener mBillingUpdatesListener;
    private final Activity mActivity;
    /**
     * A reference to BillingClient
     **/
    private BillingClient mBillingClient;
    /**
     * True if billing service is connected now.
     */
    private boolean mIsServiceConnected;
    private Set<String> mTokensToBeConsumed;

    private Set<String> mTokensToBeAcknowledged;

    private int mBillingClientResponseCode = BILLING_MANAGER_NOT_INITIALIZED;
    private boolean isQAMode;
    private static final String QA_SERVER_VERIFICATION_HOST = "https://us-central1-now-gg-payments.cloudfunctions.net/nowgg-demo-payments-backend/multi-store/verify";
    private static final String PROD_SERVER_VERIFICATION_HOST = "https://us-central1-now-gg-payments.cloudfunctions.net/nowgg-demo-payments-backend-prod/multi-store/verify";


    public BillingManager(Activity activity, final BillingUpdatesListener updatesListener) {
        Log.i(TAG, "Creating Billing client.");
        mActivity = activity;
        mBillingUpdatesListener = updatesListener;

        mBillingClient = BillingClient.newBuilder(mActivity)
                .setAppId(APP_ID)
                .setListener(this).build();

        Log.i(TAG, "Using Environment: " + BillingClient.mEnvironment.getEnvName());
        isQAMode = BillingClient.mEnvironment.getEnvName().equals(Constants.ENV_QA);

        Log.i(TAG, "Starting setup.");

        // Start setup. This is asynchronous and the specified listener will be called
        // once setup completes.
        // It also starts to report all the new purchases through onPurchasesUpdated() callback.
        startServiceConnection(new Runnable() {
            @Override
            public void run() {
                // Notifying the listener that billing client is ready
                mBillingUpdatesListener.onBillingClientSetupFinished();
                // IAB is fully set up. Now, let's get an inventory of stuff we own.
                if (mBillingClientResponseCode == BillingClient.BillingResponse.OK) {
                    Log.i(TAG, "Setup successful. Querying inventory.");
//                    queryPurchasesAsync();
                }
            }
        });
    }

    /**
     * Handle a callback that purchases were updated from the Billing library
     */
    @Override
    public void onPurchasesUpdated(int resultCode, List<Purchase> purchases) {
        Log.i(TAG, "onPurchasesUpdated() - response: " + resultCode + " purchases: " + ((purchases == null) ? null : Arrays.toString(purchases.stream().map(Purchase::getOriginalJson).toArray())));
        if (resultCode == BillingResponse.OK) {
            if (purchases == null) {
                return;
            }
            for (Purchase purchase : purchases) {
                handlePurchase(purchase);
            }
        } else {
            Log.i(TAG, "onPurchasesUpdated() - result not ok - " + Utils.getResponseString(resultCode));
            Toast.makeText(mActivity, "Error: " + Utils.getResponseString(resultCode), Toast.LENGTH_SHORT).show();
        }
    }

    /**
     * Start a purchase flow
     */
    public void initiatePurchaseFlow(final String skuId,
                                     final @SkuType String billingType,
                                     final String developerPayload

    ) {
        Log.i(TAG, "Launching in-app purchase flow. Requesting sku: " + skuId);
        Runnable purchaseFlowRequest = new Runnable() {
            @Override
            public void run() {
                BillingFlowParams purchaseParams = BillingFlowParams.newBuilder()
                        .setSku(skuId).setType(billingType).setDeveloperPayload(developerPayload).build();
                @BillingResponse int response = mBillingClient.launchBillingFlow(mActivity, purchaseParams);
                Log.i(TAG, "Launch purchase flow finished with response code: " + response);
                if (response != BillingResponse.OK) {
                    Log.i(TAG, "Unable to start purchase flow. Error response: " + Utils.getResponseString(response));
                    new Handler(Looper.getMainLooper()).post(() -> Toast.makeText(mActivity, "Error: " + Utils.getResponseString(response), Toast.LENGTH_SHORT).show());
                }
            }
        };

        executeServiceRequest(purchaseFlowRequest, false);
    }

    public Context getContext() {
        return mActivity;
    }

    /**
     * Clear the resources
     */
    public void destroy() {
        Log.i(TAG, "Destroying the manager.");

        if (mBillingClient != null && mBillingClient.isReady()) {
            Log.i(TAG, "Closing the billing client.");
            mBillingClient.endConnection();
            mBillingClient = null;
        }
    }

    public void querySkuDetailsAsync(final List<String> skuList,
                                     final SkuDetailsResponseListener listener) {
        // Creating a runnable from the request to use it inside our connection retry policy below
        Runnable queryRequest = new Runnable() {
            @Override
            public void run() {
                // Query the purchase async
                SkuDetailsParams.Builder params = SkuDetailsParams.newBuilder();
                params.setSkusList(skuList);
                mBillingClient.querySkuDetailsAsync(params.build(),
                        new SkuDetailsResponseListener() {
                            @Override
                            public void onSkuDetailsResponse(int responseCode,
                                                             List<SkuDetails> skuDetailsList) {
                                listener.onSkuDetailsResponse(responseCode, skuDetailsList);
                            }
                        });
            }
        };

        executeServiceRequest(queryRequest, true);
    }

    public void consumeAsync(final String purchaseToken) {
        // If we've already scheduled to consume this token - no action is needed (this could happen
        // if you received the token when querying purchases inside onReceive() and later from
        // onActivityResult()
        Log.i(TAG, "consumeAsync() called with: purchaseToken = [" + purchaseToken + "]");
        if (mTokensToBeConsumed == null) {
            mTokensToBeConsumed = new HashSet<>();
        } else if (mTokensToBeConsumed.contains(purchaseToken)) {
            Log.i(TAG, "Token was already scheduled to be consumed - skipping...");
            return;
        }
        mTokensToBeConsumed.add(purchaseToken);

        // Generating Consume Response listener
        final ConsumeResponseListener onConsumeListener = new ConsumeResponseListener() {
            @Override
            public void onConsumeResponse(@BillingResponse int responseCode, String purchaseToken) {
                // If billing service was disconnected, we try to reconnect 1 time
                // (feel free to introduce your retry policy here).
                Log.i(TAG, "onConsumeResponse: "+responseCode+" purchase Token "+purchaseToken);
                mBillingUpdatesListener.onConsumeFinished(purchaseToken, responseCode);
            }
        };

        // Creating a runnable from the request to use it inside our connection retry policy below
        Runnable consumeRequest = new Runnable() {
            @Override
            public void run() {
                // Consume the purchase async
                mBillingClient.consumeAsync(purchaseToken, onConsumeListener);
            }
        };

        executeServiceRequest(consumeRequest, true);
    }

    public void acknowledgeAsync(String purchaseToken) {
        // If we've already scheduled to acknowledge this token - no action is needed (this could happen
        // if you received the token when querying purchases inside onReceive() and later from
        // onActivityResult()
        if (mTokensToBeAcknowledged == null) {
            mTokensToBeAcknowledged = new HashSet<>();
        } else if (mTokensToBeAcknowledged.contains(purchaseToken)) {
            Log.i(TAG, "Token was already scheduled to be acknowdledged - skipping...");
            return;
        }
        mTokensToBeAcknowledged.add(purchaseToken);

        // Generating Acknowledge Purchase Response listener
        final AcknowledgePurchaseResponseListener acknowledgePurchaseResponseListener = new AcknowledgePurchaseResponseListener() {
            @Override
            public void onAcknowledgePurchaseResponse(@BillingResponse int responseCode) {
                mBillingUpdatesListener.onAcknowledgeFinished(responseCode);
            }
        };

// Creating a runnable from the request to use it inside our connection retry policy below
        Runnable acknowledgeRequest = new Runnable() {
            @Override
            public void run() {
                // Acknowledge the purchase async

                AcknowledgePurchaseParams acknowledgePurchaseParams = AcknowledgePurchaseParams.newBuilder()
                        .setPurchaseToken(purchaseToken)
                        .build();
                mBillingClient.acknowledgePurchase(acknowledgePurchaseParams, acknowledgePurchaseResponseListener);
            }
        };

        executeServiceRequest(acknowledgeRequest, true);
    }


    /**
     * Returns the value Billing client response code or BILLING_MANAGER_NOT_INITIALIZED if the
     * client connection response was not received yet.
     */
    public int getBillingClientResponseCode() {
        return mBillingClientResponseCode;
    }

    /**
     * Handles the purchase
     * <p>Note: Notice that for each purchase, we check if signature is valid on the client.
     * It's recommended to move this check into your backend.
     * </p>
     *
     * @param purchase Purchase to be handled
     */
    private void handlePurchase(Purchase purchase) {
        Log.i(TAG, "handlePurchase: " + purchase.getOriginalJson());

        // API call to verify server
        serverVerification(purchase.getPurchaseToken(), purchase.getSku(), mBillingClient.getStoreType(), new VerificationCallback() {
            @Override
            public void onSuccess(String response) {
                Log.i(TAG, "Inside HandlePurchase: Server verification successful callback: " + response);
                List<Purchase> mPurchases = new ArrayList<>();
                mPurchases.add(purchase);
                mBillingUpdatesListener.onPurchasesUpdated(mPurchases);
            }

            @Override
            public void onFailure(String errorMessage) {
                Log.e(TAG, "Server verification failed: " + errorMessage);
            }
        });
    }

    public interface VerificationCallback {
        void onSuccess(String response);

        void onFailure(String errorMessage);
    }

    private void serverVerification(String purchaseToken, String productId, String storeType, VerificationCallback callback) {
        OkHttpClient client = new OkHttpClient();
        String hostUrl = isQAMode ? QA_SERVER_VERIFICATION_HOST : PROD_SERVER_VERIFICATION_HOST;

        JSONObject jsonObject = new JSONObject();
        try {
            Log.i("TAG", "Server Verification Host URL: " + hostUrl);
            jsonObject.put("purchaseToken", purchaseToken);
            jsonObject.put("productId", productId);
            jsonObject.put("type", storeType);
        } catch (JSONException e) {
            callback.onFailure("Server Verification JSON creation error: " + e.getMessage());
            return;
        }

        String json = jsonObject.toString();
        Log.i(TAG, "Server verification request: " + json);
        Log.i(TAG, "Server verification host: " + hostUrl);
        RequestBody body = RequestBody.create(
                json, MediaType.parse("application/json; charset=utf-8"));

        Request request = new Request.Builder()
                .url(hostUrl)
                .post(body)
                .build();

        Handler mainHandler = new Handler(Looper.getMainLooper());

        client.newCall(request).enqueue(new okhttp3.Callback() {
            @Override
            public void onFailure(@NonNull okhttp3.Call call, @NonNull IOException e) {
                mainHandler.post(() -> callback.onFailure(e.getMessage()));
            }

            @Override
            public void onResponse(@NonNull okhttp3.Call call, @NonNull okhttp3.Response response) throws IOException {
                if (response.body() == null) {
                    Log.i(TAG, "Server verification failed: Response body is null");
                    mainHandler.post(() -> callback.onFailure("Response body is null"));
                    return;
                }
                String responseBody = response.body().string();
                if (response.isSuccessful()) {
                    Log.i(TAG, "Server verification successful: " + responseBody);
                    try {
                        JSONObject jsonResponse = new JSONObject(responseBody);
                        if (jsonResponse.getBoolean("success")) {
                            mainHandler.post(() -> callback.onSuccess(responseBody));
                        } else {
                            String error = jsonResponse.optString("codeMsg", "Unknown error");
                            mainHandler.post(() -> callback.onFailure("ServerVerification failed: " + error));
                        }
                    } catch (JSONException e) {
                        Log.e(TAG, "Server Verification JSON parsing error", e);
                        mainHandler.post(() -> callback.onFailure("Server Verification JSON parsing error: " + e.getMessage()));
                    }
                } else {
                    Log.i(TAG, "Server verification failed: " + responseBody);
                    mainHandler.post(() -> callback.onFailure("ServerVerification Error: " + responseBody));
                }
            }
        });

    }

    /**
     * Handle a result from querying of purchases and report an updated list to the listener
     */
    private void onQueryPurchasesFinished(PurchasesResult result) {
        // Have we been disposed of in the meantime? If so, or bad result code, then quit
        if (mBillingClient == null || result.getResponseCode() != BillingResponse.OK) {
            Log.w(TAG, "Billing client was null or result code (" + result.getResponseCode()
                    + ") was bad - quitting");
            return;
        }

        Log.i(TAG, "Query inventory was successful.");

        // Update the UI and purchases inventory with new list of purchases
        onPurchasesUpdated(BillingResponse.OK, result.getPurchasesList());
    }

    /**
     * Checks if subscriptions are supported for current client
     * <p>Note: This method does not automatically retry for RESULT_SERVICE_DISCONNECTED.
     * It is only used in unit tests and after queryPurchases execution, which already has
     * a retry-mechanism implemented.
     * </p>
     */
    public boolean areSubscriptionsSupported() {
        int responseCode = mBillingClient.isFeatureSupported(FeatureType.SUBSCRIPTIONS);
        if (responseCode != BillingResponse.OK) {
            Log.w(TAG, "areSubscriptionsSupported() got an error response: " + responseCode);
        }
        return responseCode == BillingResponse.OK;
    }

    public void queryPurchasesAsync() {
        if (mBillingClient.isReady() == false) {
            Log.w(TAG, "queryPurchasesAsync() - client is not ready to query purchases.");
            return;
        } else {
            Log.i(TAG, "queryPurchasesAsync() - client is ready to query purchases.");
        }

        QueryPurchasesParams params = QueryPurchasesParams.newBuilder().setProductType(BillingClient.ProductType.ALL).build();
        mBillingClient.queryPurchasesAsync(params, new PurchasesResponseListener() {
            @Override
            public void onQueryPurchasesResponse(BillingResult billingResult, List<Purchase> list) {
                if (billingResult.getResponseCode() == BillingResponse.OK) {
                    onPurchasesUpdated(billingResult.getResponseCode(), list);
                } else {
                    BillingHelper.logWarn(TAG, "queryPurchasesAsync() got an error response code: " + billingResult.getResponseCode() + ", error :" + Utils.getResponseString(billingResult.getResponseCode()));
                }
            }
        });
    }

    public void queryProductDetailsAsync(final List<QueryProductDetailsParams.Product> productList,
                                         final ProductDetailsResponseListener listener) {
        Log.i(TAG, "queryProductDetailsAsync() called with: productList = [" + productList + "]");
        // Creating a runnable from the request to use it inside our connection retry policy below
        Runnable queryRequest = new Runnable() {
            @Override
            public void run() {
                // Query the purchase async
                QueryProductDetailsParams.Builder params = QueryProductDetailsParams.newBuilder();
                params.setProductList(productList);
                mBillingClient.queryProductDetailsAsync(params.build(), new ProductDetailsResponseListener() {
                    @Override
                    public void onProductDetailsResponse(int billingResult, List<ProductDetails> productDetailsList) {
                        listener.onProductDetailsResponse(billingResult, productDetailsList);
                    }
                });
            }
        };

        executeServiceRequest(queryRequest, true);
    }

    public void startServiceConnection(final Runnable executeOnSuccess) {
        mBillingClient.startConnection(new BillingClientStateListener() {
            @Override
            public void onBillingSetupFinished(@BillingResponse int billingResponseCode) {
                Log.i(TAG, "Setup finished. Response code: " + billingResponseCode);

                mBillingClientResponseCode = billingResponseCode;

                if (billingResponseCode == BillingResponse.OK) {
                    mIsServiceConnected = true;
                    if (executeOnSuccess != null) {
                        executeOnSuccess.run();
                    }
                }

                queryPurchasesAsync();

            }

            @Override
            public void onBillingServiceDisconnected() {
                mIsServiceConnected = false;
            }
        });
    }

    private void executeServiceRequest(Runnable runnable, Boolean isUiThread) {
        if (mIsServiceConnected) {
            if (isUiThread) {
                runnable.run();
            } else {
                new Thread(runnable).start();
            }
        } else {
            // If billing service was disconnected, we try to reconnect 1 time.
            // (feel free to introduce your retry policy here).
            startServiceConnection(runnable);
        }
    }

    /**
     * Verifies that the purchase was signed correctly for this developer's public key.
     * <p>Note: It's strongly recommended to perform such check on your backend since hackers can
     * replace this method with "constant true" if they decompile/rebuild your app.
     * </p>
     */
    private boolean verifyValidSignature(String signedData, String signature) {
        try {
            if (isQAMode) {
                return Security.verifyPurchase(BASE_64_ENCODED_PUBLIC_KEY_QA, signedData, signature, mBillingClient.getStoreType());

            } else {
                return Security.verifyPurchase(BASE_64_ENCODED_PUBLIC_KEY, signedData, signature, mBillingClient.getStoreType());
            }
        } catch (IOException e) {
            Log.e(TAG, "Got an exception trying to validate a purchase: " + e);
            return false;
        }
    }

    /**
     * Listener to the updates that happen when purchases list was updated or consumption of the
     * item was finished
     */
    public interface BillingUpdatesListener {
        void onBillingClientSetupFinished();

        void onConsumeFinished(String token, @BillingResponse int result);

        void onPurchasesUpdated(List<Purchase> purchases);

        void onAcknowledgeFinished(@BillingClient.BillingResponse int result);

    }
}
