package gg.now.nowggsdkdemo.payments.skulist;

import static gg.now.billingclient.api.BillingClient.BillingResponse;

import android.content.DialogInterface;
import android.content.res.Resources;
import android.os.Bundle;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.TextView;
import android.widget.Toast;

import androidx.annotation.Nullable;
import androidx.annotation.StringRes;
import androidx.appcompat.widget.Toolbar;
import androidx.fragment.app.DialogFragment;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

import java.util.ArrayList;
import java.util.List;

import gg.now.billingclient.api.BillingClient.SkuType;
import gg.now.billingclient.api.ProductDetails;
import gg.now.billingclient.api.ProductDetailsResponseListener;
import gg.now.billingclient.api.QueryProductDetailsParams;
import gg.now.nowggsdkdemo.R;
import gg.now.nowggsdkdemo.payments.billing.BillingProvider;
import gg.now.nowggsdkdemo.payments.skulist.row.SkuRowData;
import gg.now.nowggsdkdemo.payments.skulist.row.UiManager;

/**
 * Displays a screen with various in-app purchase and subscription options
 */
public class AcquireFragment extends DialogFragment {
    private static final String TAG = "AcquireFragment";
    static boolean loadFinished;
    private RecyclerView mRecyclerView;
    private SkusAdapter mAdapter;
    private View mLoadingView;
    private TextView mErrorTextView;
    private BillingProvider mBillingProvider;

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setStyle(DialogFragment.STYLE_NORMAL, R.style.Theme_nowggsdkdemo);
    }

    @Nullable
    @Override
    public View onCreateView(LayoutInflater inflater, @Nullable ViewGroup container,
                             @Nullable Bundle savedInstanceState) {
        View root = inflater.inflate(R.layout.acquire_fragment, container, false);
        mErrorTextView = root.findViewById(R.id.error_textview);
        mRecyclerView = root.findViewById(R.id.list);
        mLoadingView = root.findViewById(R.id.screen_wait);
        if (mBillingProvider != null) {
            handleManagerAndUiReady();
        }
        // Setup a toolbar for this fragment
        Toolbar toolbar = root.findViewById(R.id.toolbar);
        toolbar.setNavigationIcon(R.drawable.ic_arrow_up);
        toolbar.setNavigationOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                dismiss();
            }
        });
        toolbar.setTitle(R.string.button_purchase);
        return root;
    }

    /**
     * Refreshes this fragment's UI
     */
    public void refreshUI() {
        Log.i(TAG, "Looks like purchases list might have been updated - refreshing the UI");
        if (mAdapter != null) {
            mAdapter.notifyDataSetChanged();
        }
    }

    /**
     * Notifies the fragment that billing manager is ready and provides a BillingProviders
     * instance to access it
     */
    public void onManagerReady(BillingProvider billingProvider) {
        mBillingProvider = billingProvider;
        if (mRecyclerView != null) {
            handleManagerAndUiReady();
        }
    }

    /**
     * Enables or disables "please wait" screen.
     */
    private void setWaitScreen(boolean set) {
        mRecyclerView.setVisibility(set ? View.GONE : View.VISIBLE);
        mLoadingView.setVisibility(set ? View.VISIBLE : View.GONE);
    }

    /**
     * Executes query for SKU details at the background thread
     */
    private void handleManagerAndUiReady() {
        // If Billing Manager was successfully initialized - start querying for SKUs
        setWaitScreen(true);
        querySkuDetails();
    }

    private void displayAnErrorIfNeeded() {
        if (getActivity() == null || getActivity().isFinishing()) {
            Log.i(TAG, "No need to show an error - activity is finishing already");
            return;
        }

        mLoadingView.setVisibility(View.GONE);
        mErrorTextView.setVisibility(View.VISIBLE);
        int billingResponseCode = mBillingProvider.getBillingManager()
                .getBillingClientResponseCode();

        switch (billingResponseCode) {
            case BillingResponse.OK:
                // If manager was connected successfully, then show no SKUs error
                mErrorTextView.setText(getText(R.string.error_no_skus));
                break;
            case BillingResponse.BILLING_UNAVAILABLE:
                mErrorTextView.setText(getText(R.string.error_billing_unavailable));
                break;
            default:
                mErrorTextView.setText(getText(R.string.error_billing_default));
        }

    }

    /**
     * Queries for in-app SKU details and updates an adapter with new data
     */
    private void querySkuDetails() {
        long startTime = System.currentTimeMillis();

        Log.i(TAG, "querySkuDetails() got subscriptions and inApp SKU details lists for: "
                + (System.currentTimeMillis() - startTime) + "ms");

        if (getActivity() != null && !getActivity().isFinishing()) {
            final List<SkuRowData> dataList = new ArrayList<>();
            mAdapter = new SkusAdapter();
            final UiManager uiManager = createUiManager(mAdapter, mBillingProvider);
            mAdapter.setUiManager(uiManager);

            //Fill the in-app items rows below
            List<String> inAppSkus = uiManager.getDelegatesFactory().getSkuList(SkuType.INAPP);
            List<String> inAppSkusSubs = uiManager.getDelegatesFactory().getSkuList(SkuType.SUBS);
            inAppSkus.addAll(inAppSkusSubs);
            addSkuRows(dataList, inAppSkus, new Runnable() {
                @Override
                public void run() {
                    loadFinished = true;
                }
            });

        }
    }

    private void addSkuRows(final List<SkuRowData> inList, List<String> skusList, final Runnable executeWhenFinished) {

        loadFinished = false;
        List<QueryProductDetailsParams.Product> productList = new ArrayList<>();
        for (String skuId : skusList) {
            productList.add(QueryProductDetailsParams.Product.newBuilder().setProductId(skuId).build());
        }

        mBillingProvider.getBillingManager().queryProductDetailsAsync(productList, new ProductDetailsResponseListener() {
            @Override
            public void onProductDetailsResponse(int responseCode, List<ProductDetails> productDetailsList) {
                if (responseCode != BillingResponse.OK) {
                    Log.w(TAG, "Unsuccessful query"
                            + ". Error code: " + responseCode);
                } else if (productDetailsList != null
                        && productDetailsList.size() > 0) {
                    // If we successfully got SKUs, add a header in front of the row
                    String currentHeaderType = productDetailsList.get(0).getProductType();
                    @StringRes int stringRes = SkuType.INAPP.equals(currentHeaderType)
                            ? R.string.header_inapp : R.string.header_subscriptions;
                    inList.add(new SkuRowData(getString(stringRes)));
                    // Then fill all the other rows
                    for (ProductDetails details : productDetailsList) {
                        Log.i(TAG, "Adding sku: " + details);
                        if (!currentHeaderType.equals(details.getProductType())) {
                            currentHeaderType = details.getProductType();
                            @StringRes int stringHeaderRes = SkuType.INAPP.equals(currentHeaderType)
                                    ? R.string.header_inapp : R.string.header_subscriptions;
                            inList.add(new SkuRowData(getString(stringHeaderRes)));
                        }
                        inList.add(new SkuRowData(details, SkusAdapter.TYPE_NORMAL,
                                details.getProductType()));
                    }

                    if (inList.size() == 0) {
                        displayAnErrorIfNeeded();
                    } else {
                        if (mRecyclerView.getAdapter() == null) {
                            mRecyclerView.setAdapter(mAdapter);
                            Resources res = getContext().getResources();
                            mRecyclerView.addItemDecoration(new CardsWithHeadersDecoration(
                                    mAdapter, (int) res.getDimension(R.dimen.DIMEN_8DP),
                                    (int) res.getDimension(R.dimen.row_gap)));
                            mRecyclerView.setLayoutManager(
                                    new LinearLayoutManager(getContext()));
                        }

                        mAdapter.updateData(inList);
                        setWaitScreen(false);
                    }
                } else {
                    // Handle empty state
                    displayAnErrorIfNeeded();
                }

                if (executeWhenFinished != null) {
                    executeWhenFinished.run();
                }
            }
        });
//        mBillingProvider.getBillingManager().querySkuDetailsAsync(billingType, skusList,
//                new SkuDetailsResponseListener() {
//                    @Override
//                    public void onSkuDetailsResponse(int responseCode, List<SkuDetails> skuDetailsList) {
//
//                        if (responseCode != BillingResponse.OK) {
//                            Log.w(TAG, "Unsuccessful query for type: " + billingType
//                                    + ". Error code: " + responseCode);
//                        } else if (skuDetailsList != null
//                                && skuDetailsList.size() > 0) {
//                            // If we successfully got SKUs, add a header in front of the row
//                            @StringRes int stringRes = SkuType.INAPP.equals(billingType)
//                                    ? R.string.header_inapp : R.string.header_subscriptions;
//                            inList.add(new SkuRowData(getString(stringRes)));
//                            // Then fill all the other rows
//                            for (SkuDetails details : skuDetailsList) {
//                                Log.i(TAG, "Adding sku: " + details);
//                                inList.add(new SkuRowData(details, SkusAdapter.TYPE_NORMAL,
//                                        billingType));
//                            }
//
//                            if (inList.size() == 0) {
//                                displayAnErrorIfNeeded();
//                            } else {
//                                if (mRecyclerView.getAdapter() == null) {
//                                    mRecyclerView.setAdapter(mAdapter);
//                                    Resources res = getContext().getResources();
//                                    mRecyclerView.addItemDecoration(new CardsWithHeadersDecoration(
//                                            mAdapter, (int) res.getDimension(R.dimen.DIMEN_8DP),
//                                            (int) res.getDimension(R.dimen.row_gap)));
//                                    mRecyclerView.setLayoutManager(
//                                            new LinearLayoutManager(getContext()));
//                                }
//
//                                mAdapter.updateData(inList);
//                                setWaitScreen(false);
//                            }
//                        } else {
//                            // Handle empty state
//                            displayAnErrorIfNeeded();
//                        }
//
//                        if (executeWhenFinished != null) {
//                            executeWhenFinished.run();
//                        }
//                    }
//                });
    }

    protected UiManager createUiManager(SkusAdapter adapter, BillingProvider provider) {
        return new UiManager(adapter, provider);
    }

    @Override
    public void onResume() {
        super.onResume();
        getDialog().setOnKeyListener(new DialogInterface.OnKeyListener() {
            @Override
            public boolean onKey(DialogInterface dialog, int keyCode, android.view.KeyEvent event) {
                if ((keyCode == android.view.KeyEvent.KEYCODE_BACK)) {
                    // To dismiss the fragment when the back-button is pressed.
                    dismiss();
                    return true;
                }
                // Otherwise, do nothing else
                else {
                    return false;
                }
            }
        });

    }

    @Override
    public void dismiss() {
        if (loadFinished) {
            super.dismiss();
        } else {
            Toast.makeText(getActivity(), "Waiting for data loading.", Toast.LENGTH_SHORT).show();
        }
    }
}
