using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using UnityEngine.Purchasing;
using UnityEngine.Purchasing.Extension;

namespace NowGG.Sdk
{
    public class NowGGStore : IStore
    {
        private IStoreCallback callback;
        private string paymentId;
        //private string inGameID;
        private string lastUsedProductId;

        // Use this when implementation Subscriptions
        //private Dictionary<string, bool> acknowledgedTransactions = new Dictionary<string, bool>();

        private void OnNowGGSdkInitSuccess()
        {
            NowGGLogger.Log("NowggSdk initialization success");
            List<ProductDescription> productsFetched = new List<ProductDescription>();
            if (NowGGPaymentsSdkManager.Instance.products.Length > 0)
            {
                for (int i = 0; i < NowGGPaymentsSdkManager.Instance.products.Length; i++)
                {
                    var initialProduct = NowGGPaymentsSdkManager.Instance.products[i];
                    var productMetadata = new ProductMetadata(initialProduct.price,
                        initialProduct.title, initialProduct.description,
                        initialProduct.priceCurrencyCode,
                        decimal.Parse(initialProduct.priceAmountMicros) / 1000000);

                    ProductDescription finalProduct = new(initialProduct.productId, productMetadata);
                    productsFetched.Add(finalProduct);
                }
            }
            callback.OnProductsRetrieved(productsFetched);
        }

        private void OnNowGGSdkInitFailed(string error)
        {
            NowGGLogger.Log($"NowggSdk initialization failed with error: {error}");
            callback.OnSetupFailed(InitializationFailureReason.PurchasingUnavailable, error);
        }

        PurchaseProcessingResult OnNowGGSdkPurchaseProduct(PurchasedProduct purchasedProduct)
        {
            try
            {
                NowGGLogger.Log($"Purchased product: {purchasedProduct.productId}");

                //acknowledgedTransactions.Add(purchasedProduct.purchaseToken, purchasedProduct.isAcknowledged);

                Dictionary<string, object> receiptDict = new Dictionary<string, object>();

                // Required for purchase validation
                receiptDict.Add("productId", purchasedProduct.productId);
                receiptDict.Add("purchaseToken", purchasedProduct.purchaseToken);
                receiptDict.Add("json", purchasedProduct.originalJson);
                receiptDict.Add("signature", purchasedProduct.signature);

                callback.OnPurchaseSucceeded(purchasedProduct.productId,
                    MiniJson.JsonEncode(receiptDict), purchasedProduct.purchaseToken);
            }
            catch (Exception e)
            {
                NowGGLogger.Log($"Error in IStore module: {e.ToString()}");
            }

            return PurchaseProcessingResult.Pending;
        }

        public void OnNowGGSdkPurchaseFailed(int errorCode, string errorMessage)
        {
            PurchaseFailureReason failureReason = PurchaseFailureReason.Unknown;
            if (errorCode == 1)
            {
                failureReason = PurchaseFailureReason.UserCancelled;
            }
            else if (errorCode == 2 || errorCode == 3)
            {
                failureReason = PurchaseFailureReason.PurchasingUnavailable;
            }
            else if (errorCode == 4)
            {
                failureReason = PurchaseFailureReason.ProductUnavailable;
            }
            else if (errorCode == 9)
            {
                failureReason = PurchaseFailureReason.PaymentDeclined;
            }

            NowGGLogger.Log($"Purchase failed for product: {lastUsedProductId} Reason: {errorMessage}");
            callback.OnPurchaseFailed(new PurchaseFailureDescription(lastUsedProductId,
                failureReason, errorMessage));
        }

        public void Initialize(IStoreCallback callback)
        {
            NowGGLogger.Log("Initializing v2.0.0 of the NowGGStore");
            paymentId = NowGGPaymentsSdkManager.Instance.PaymentId ?? "";
            //inGameID = NowGGPaymentsSdkManager.Instance.InGameId ?? "";

            if (paymentId.Length < 1)
            {
                var error = "PaymentId not set in NowGGPaymentsSdkManager";
                NowGGLogger.LogError(error);
                callback.OnSetupFailed(InitializationFailureReason.AppNotKnown, error);
                return;
            }

            NowGGLogger.Log($"Setting PaymentId: {paymentId}");

            NowGGPaymentsSdkManager.Instance.OnInitSuccess += OnNowGGSdkInitSuccess;
            NowGGPaymentsSdkManager.Instance.OnInitFailed += OnNowGGSdkInitFailed;
            NowGGPaymentsSdkManager.Instance.OnPurchaseCompleted += OnNowGGSdkPurchaseProduct;
            NowGGPaymentsSdkManager.Instance.OnPurchaseFailed += OnNowGGSdkPurchaseFailed;

            this.callback = callback;
        }

        public void RetrieveProducts(ReadOnlyCollection<ProductDefinition> products)
        {
            NowGGPaymentsSdkManager.Instance.IsInitialized = false;
            NowGGProductBuilder.Instance.Clear();

            for (int i = 0; i < products.Count; i++)
            {
                var product = products[i];

                if (product.type != UnityEngine.Purchasing.ProductType.Consumable)
                {
                    var error = $"Nowgg currently doesn't support product types other than consumable. Purchasing will not work for product: {product.id}.";
                    NowGGLogger.LogError(error);
                }

                NowGGProductBuilder.Instance.AddProduct(product.id, NowGG.Sdk.ProductType.Consumable);
            }

                NowGGLogger.Log($"inGameId is null. Initializing iap with nowgg login flow");
                NowGGPaymentsSdkManager.Instance.InitializeIap(paymentId);
        }

        public void Purchase(ProductDefinition product, string developerPayload)
        {
            NowGGLogger.Log($"Purchasing {product.id}");
            lastUsedProductId = product.id;
            NowGGPaymentsSdkManager.Instance.PurchaseProduct(product.id, developerPayload);
        }

        public void FinishTransaction(ProductDefinition product, string transactionId)
        {
            NowGGLogger.Log($"Finishing transaction for {product.id}");
            NowGGPaymentsSdkManager.Instance.ConfirmPendingPurchase(transactionId,
                NowGGProductBuilder.Instance.GetProductType(product.id));
        }
    }
}