using NowGG.Sdk;
using System;
using System.Collections;
using TMPro;
using UnityEngine;

public class PaymentsHandler : MonoBehaviour
{
    private string PAYMENTS_APP_ID = "10156";

    [Header("IAP Products")]
    public TMP_Text TextCoin1Price;
    public TMP_Text TextCoin2Price;
    public TMP_Text TextCoin3Price;

    [Header("Subscription 1")]
    public TMP_Text TextSub1Price;
    public GameObject GoBtnBuySub1;
    public GameObject GoContainerSub1Status;
    public TMP_Text TextSub1Status;
    public TMP_Text TextSub1Validity;

    [Header("Subscription 2")]
    public TMP_Text TextSub2Price;
    public GameObject GoBtnBuySub2;
    public GameObject GoContainerSub2Status;
    public TMP_Text TextSub2Status;
    public TMP_Text TextSub2Validity;

    [Header("Misc")]
    public TMP_Text TextStatus;
    public TMP_Text TextCoinsCount;
    public GameObject GoBtnInitPayments;
    public GameObject GoContainerIapItems;
    public GameObject GoLoader;

    private int coinsCount = 500;
    private int previousCoinsCount = 0;

    private string coin1000 = "coin_1000";
    private string coin2000 = "coin_2000";
    private string coin3000 = "coin_3000";
    private string weeklySub = "weeklySub";
    private string monthlySub = "monthlySub";

    private void Start()
    {
        TextCoinsCount.text = $"Coins: {coinsCount}";
    }

    private void SetupNowggPayments()
    {
        if (Globals.IsPaymentsInitialized)
        {
            Utils.Log("Payments already initialized. Showing IAP itesm");
            ShowSkuItems();
            return;
        }

        Utils.Log("Setting up nowgg payments");
        GoLoader.SetActive(true);

        NowGGPaymentsSdkManager.Instance.OnInitSuccess += OnInitSuccess;
        NowGGPaymentsSdkManager.Instance.OnInitFailed += OnInitFailed;
        NowGGPaymentsSdkManager.Instance.OnPurchaseCompleted += OnPurchaseProduct;
        NowGGPaymentsSdkManager.Instance.OnPurchaseFailed += OnPurchaseFailed;

        AddProducts();

        Utils.Log("Initializing nowgg payments");
        NowGGPaymentsSdkManager.Instance.InitializeIap(PAYMENTS_APP_ID);
    }

    private void AddProducts()
    {
        NowGGProductBuilder.Instance.AddProduct(coin1000, ProductType.Consumable);
        NowGGProductBuilder.Instance.AddProduct(coin2000, ProductType.Consumable);
        NowGGProductBuilder.Instance.AddProduct(coin3000, ProductType.Consumable);

        NowGGProductBuilder.Instance.AddProduct(weeklySub, ProductType.Subscription);
        NowGGProductBuilder.Instance.AddProduct(monthlySub, ProductType.Subscription);
    }

    private void OnInitSuccess()
    {
        GoLoader.SetActive(false);
        Globals.IsPaymentsInitialized = true;
        TextStatus.text = "Payments initialized";
        Utils.Log("Payments initialized. Showing IAP items");
        Invoke(nameof(ShowSkuItems), 0.1f);
    }

    private void OnInitFailed(string error)
    {
        GoLoader.SetActive(false);
        Utils.Log("Payments init failed. Error: " + error);
        TextStatus.text = "Payments init failed: " + error;
    }

    public void OnPurchaseFailed(int errorCode, string errorMessage)
    {
        GoLoader.SetActive(false);
        Utils.Log("Purchase failed. Error: " + errorMessage);
        TextStatus.text = "Purchase failed: " + errorMessage;
    }

    PurchaseProcessingResult OnPurchaseProduct(PurchasedProduct purchasedProduct)
    {
        Utils.Log($"IAP item purchased: {purchasedProduct.productId}");
        Utils.Log($"Developer payload: {purchasedProduct.developerPayload}");

        TextStatus.text = $"Successfully bought={purchasedProduct.productId}, devPayload={purchasedProduct.developerPayload}";

        previousCoinsCount = coinsCount;

        // A consumable product has been purchased by this user.
        if (String.Equals(purchasedProduct.productId, coin1000, StringComparison.Ordinal))
        {
            Utils.Log(string.Format("OnPurchaseProduct: PASS. Product: '{0}'", purchasedProduct.productId));
            // The consumable item has been successfully purchased, add 100 coins to the player's in-game currency.
            coinsCount += 1000;
            UpdateCoinsCount();
        }
        else if (String.Equals(purchasedProduct.productId, coin2000, StringComparison.Ordinal))
        {
            Utils.Log(string.Format("OnPurchaseProduct: PASS. Product: '{0}'", purchasedProduct.productId));
            // The consumable item has been successfully purchased, add 200 coins to the player's in-game currency.
            coinsCount += 2000;
            UpdateCoinsCount();
        }
        else if (String.Equals(purchasedProduct.productId, coin3000, StringComparison.Ordinal))
        {
            Utils.Log(string.Format("OnPurchaseProduct: PASS. Product: '{0}'", purchasedProduct.productId));
            // The consumable item has been successfully purchased, add 500 coins to the player's in-game currency.
            coinsCount += 3000;
            UpdateCoinsCount();
        }
        else if (String.Equals(purchasedProduct.productId, weeklySub, StringComparison.Ordinal))
        {
            MainThreadDispatcher.Enqueue(() => StartCoroutine(UpdateWeeklyPassStatus(purchasedProduct)));
        }
        else if (String.Equals(purchasedProduct.productId, monthlySub, StringComparison.Ordinal))
        {
            MainThreadDispatcher.Enqueue(() => StartCoroutine(UpdateMonthlyPassStatus(purchasedProduct)));
        }

        // Return a flag indicating whether this product has completely been received, or if the application needs 
        // to be reminded of this purchase at the next app launch.
        // return PurchaseProcessingResult.Pending if you want to be reminded of this purchase on the next launch of your game.
        // return PurchaseProcessingResult.Complete to specify that this is a one time purchase and you have allotted the equivalent in-game item to the user.
        return PurchaseProcessingResult.Complete;
    }

    private void UpdateCoinsCount()
    {
        StartCoroutine(CountTo(previousCoinsCount, coinsCount));
    }

    IEnumerator UpdateWeeklyPassStatus(PurchasedProduct product)
    {
        Utils.Log($"Subscription product: {product.productId} status: {product.subscriptionStatus}");

        if (product.subscriptionStatus != "ACTIVE")
        {
            // Handle expired product
            yield break;
        }

        DateTime expiryTime = DateTime.UnixEpoch.AddMilliseconds(product.expiryTimeMillis);

        GoBtnBuySub1.SetActive(false); ;
        GoContainerSub1Status.SetActive(true);

        TextSub1Status.text = $"Status: {product.subscriptionStatus}";
        TextSub1Validity.text = $"Valid till: {expiryTime.ToString("d-MMM-yy")}";

    }

    IEnumerator UpdateMonthlyPassStatus(PurchasedProduct product)
    {
        Utils.Log($"Subscription product: {product.productId} status: {product.subscriptionStatus}");

        if (product.subscriptionStatus != "ACTIVE")
        {
            // Handle expired product
            yield return null;
        }

        DateTime expiryTime = DateTime.UnixEpoch.AddMilliseconds(product.expiryTimeMillis);


        GoBtnBuySub2.SetActive(false);
        GoContainerSub2Status.SetActive(true);

        TextSub2Status.text = $"Status: {product.subscriptionStatus}";
        TextSub2Validity.text = $"Valid till: {expiryTime.ToString("d-MMM-yy")}";
    }

    private void ShowSkuItems()
    {
        GoBtnInitPayments.SetActive(false);
        GoContainerIapItems.SetActive(true);

        TextCoin1Price.text = NowGGPaymentsSdkManager.Instance.GetProductWithID(coin1000).price;
        TextCoin2Price.text = NowGGPaymentsSdkManager.Instance.GetProductWithID(coin2000).price;
        TextCoin3Price.text = NowGGPaymentsSdkManager.Instance.GetProductWithID(coin3000).price;

        TextSub1Price.text = NowGGPaymentsSdkManager.Instance.GetProductWithID(weeklySub).price;
        TextSub2Price.text = NowGGPaymentsSdkManager.Instance.GetProductWithID(monthlySub).price;
    }

    IEnumerator CountTo(int from, int to)
    {
        float animationDuration = 2.0f;
        int start = from;

        for (float timer = 0; timer < animationDuration; timer += Time.deltaTime)
        {
            float progress = timer / animationDuration;
            int numberToDisplay = (int)Mathf.Lerp(start, to, progress);

            TextCoinsCount.text = $"Coins: {numberToDisplay}";

            yield return null;
        }

        TextCoinsCount.text = $"Coins: {to}";

        yield return new WaitForEndOfFrame();
    }

    public void PurchaseProduct(string productId)
    {
        string developerPayload = "userId=123,level=5";
        NowGGPaymentsSdkManager.Instance.PurchaseProduct(productId, developerPayload);
    }

    public void BtnBackClicked()
    {
        Utils.LoadScene(Constants.SceneName.MainMenu);
    }

    public void BtnInitNowggPaymentsClicked()
    {
        SetupNowggPayments();
    }

}
