using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using TMPro;
using UnityEngine;
using UnityEngine.Purchasing;
using UnityEngine.Purchasing.Extension;

public class PaymentsHandlerQuick : MonoBehaviour, IDetailedStoreListener
{
    private string PAYMENTS_APP_ID = "10156";

    [Header("IAP Products")]
    public TMP_Text TextCoin1Price;
    public TMP_Text TextCoin2Price;
    public TMP_Text TextCoin3Price;

    [Header("Misc")]
    public TMP_Text TextStatus;
    public TMP_Text TextCoinsCount;
    public GameObject GoBtnInitPayments;
    public GameObject GoContainerIapItems;
    public GameObject GoLoader;

    ConfigurationBuilder builder;

    IStoreController IStoreController;
    IExtensionProvider IStoreExtensions;

    private int coinsCount = 500;
    private int previousCoinsCount = 0;
    private string coin1000 = "coin_1000";
    private string coin2000 = "coin_2000";
    private string coin3000 = "coin_3000";

    private void Start()
    {
        TextCoinsCount.text = $"Coins: {coinsCount}";
    }

    private void SetupNowggPayments()
    {
        if (Globals.IsPaymentsInitialized)
        {
            Utils.Log("Payments already initialized. Showing IAP items");
            ShowSkuItems();
            return;
        }

        GoLoader.SetActive(true);
        Utils.Log("Adding products to nowgg payments");

        NowGG.Sdk.NowGGPaymentsSdkManager.Instance.PaymentId = PAYMENTS_APP_ID;

        AddProducts();

        Utils.Log("Initializing nowgg payments");
        UnityPurchasing.Initialize(this, builder);
    }

    private void AddProducts()
    {

        builder = ConfigurationBuilder.Instance(NowGG.Sdk.NowGGPurchasingModule.Instance());

        builder.AddProduct(coin1000, ProductType.Consumable);
        builder.AddProduct(coin2000, ProductType.Consumable);
        builder.AddProduct(coin3000, ProductType.Consumable);

    }

    public void OnPurchaseFailed(Product product, PurchaseFailureDescription failureDescription)
    {
        OnPurchaseFailed((int)failureDescription.reason, failureDescription.reason.ToString());
    }

    public void OnPurchaseFailed(Product product, PurchaseFailureReason failureReason)
    {
        OnPurchaseFailed((int)failureReason, failureReason.ToString());
    }

    public void OnInitializeFailed(InitializationFailureReason error)
    {
        OnInitFailed(error.ToString());
    }

    public void OnInitializeFailed(InitializationFailureReason error, string message)
    {
        OnInitFailed($"Message::: {message} Reason::: {error}");
    }

    public PurchaseProcessingResult ProcessPurchase(PurchaseEventArgs purchaseEvent)
    {
        return OnPurchaseProduct(purchaseEvent);
    }

    public void OnInitialized(IStoreController controller, IExtensionProvider extensions)
    {
        OnInitSuccess(controller, extensions);
    }

    private void OnInitSuccess(IStoreController controller, IExtensionProvider extensions)
    {
        GoLoader.SetActive(false);
        Globals.IsPaymentsInitialized = true;
        TextStatus.text = "Payments initialized";
        Utils.Log("Payments initialized. Showing IAP items");
        IStoreController = controller;
        IStoreExtensions = extensions;
        Invoke(nameof(ShowSkuItems), 0.1f);
    }

    private void OnInitFailed(string error)
    {
        GoLoader.SetActive(false);
        Utils.Log("Payments init failed. Error: " + error);
        TextStatus.text = "Payments init failed: " + error;
    }

    public void OnPurchaseFailed(int errorCode, string errorMessage)
    {
        GoLoader.SetActive(false);
        Utils.Log("Purchase failed. Error: " + errorMessage);
        TextStatus.text = "Purchase failed: " + errorMessage;
    }

    PurchaseProcessingResult OnPurchaseProduct(PurchaseEventArgs purchaseEvent)
    {
        String purchasedProductId = purchaseEvent.purchasedProduct.definition.id;
        Utils.Log($"IAP item purchased: {purchasedProductId}");
        Utils.Log($"Receipt: {purchaseEvent.purchasedProduct.receipt}");

        var receiptDict = (Dictionary<string, object>)MiniJson.JsonDecode(purchaseEvent.purchasedProduct.receipt);
        string payloadString = (string)receiptDict["Payload"];
        var payloadDict = (Dictionary<string, object>)MiniJson.JsonDecode(payloadString);
        //Stuff from json object
        string jsonString = (string)payloadDict["json"];
        var jsonDict = (Dictionary<string, object>)MiniJson.JsonDecode(jsonString);

        TextStatus.text = $"productID={purchasedProductId}, devPayload={jsonDict["developerPayload"]}";

        previousCoinsCount = coinsCount;

        // A consumable product has been purchased by this user.
        if (String.Equals(purchasedProductId, coin1000, StringComparison.Ordinal))
        {
            Utils.Log(string.Format("OnPurchaseProduct: PASS. Product: '{0}'", purchasedProductId));
            // The consumable item has been successfully purchased, add 100 coins to the player's in-game currency.
            ServerVerificationNew.mInstance.purchaseServerVerification(purchaseEvent,()=>
            {
                if (this != null && gameObject.activeInHierarchy)
                {
                    AddCoins(1000); // or 2000/3000 as appropriate
                }
            });
            return PurchaseProcessingResult.Pending;
           
        }
        else if (String.Equals(purchasedProductId, coin2000, StringComparison.Ordinal))
        {
            Utils.Log(string.Format("OnPurchaseProduct: PASS. Product: '{0}'", purchasedProductId));
            // The consumable item has been successfully purchased, add 200 coins to the player's in-game currency.
            ServerVerificationNew.mInstance.purchaseServerVerification(purchaseEvent, () =>
            {
                if (this != null && gameObject.activeInHierarchy)
                {
                    AddCoins(2000); // or 2000/3000 as appropriate
                }
            });
            return PurchaseProcessingResult.Pending;
        }
        else if (String.Equals(purchasedProductId, coin3000, StringComparison.Ordinal))
        {
            Utils.Log(string.Format("OnPurchaseProduct: PASS. Product: '{0}'", purchasedProductId));
            // The consumable item has been successfully purchased, add 500 coins to the player's in-game currency.
            ServerVerificationNew.mInstance.purchaseServerVerification(purchaseEvent, () =>
            {
                if (this != null && gameObject.activeInHierarchy)
                {
                    AddCoins(3000); // or 2000/3000 as appropriate
                }
            });
            return PurchaseProcessingResult.Pending;
        }

        // Return a flag indicating whether this product has completely been received, or if the application needs 
        // to be reminded of this purchase at the next app launch.
        // return PurchaseProcessingResult.Pending if you want to be reminded of this purchase on the next launch of your game.
        // return PurchaseProcessingResult.Complete to specify that this is a one time purchase and you have allotted the equivalent in-game item to the user.
        return PurchaseProcessingResult.Complete;
    }

    private void AddCoins(int coins)
    {
        coinsCount += coins;
        UpdateCoinsCount();
    }

    private void UpdateCoinsCount()
    {
        StartCoroutine(CountTo(previousCoinsCount, coinsCount));
    }

    private void ShowSkuItems()
    {
        GoBtnInitPayments.SetActive(false);
        GoContainerIapItems.SetActive(true);

        Product[] fetchedProducts = IStoreController.products.all;
        Utils.Log($"fetched products length: {fetchedProducts.Length}");

        for (int i = 0; i < fetchedProducts.Length; i++)
        {
            Product product = fetchedProducts[i];
            Utils.Log($"fetched products length: {product.definition.id}");
            if (Equals(product.definition.id, coin1000))
            {
                TextCoin1Price.text = product.metadata.localizedPriceString;
            }
            else if (Equals(product.definition.id, coin2000))
            {
                TextCoin2Price.text = product.metadata.localizedPriceString;
            }
            else if (Equals(product.definition.id, coin3000))
            {
                TextCoin3Price.text = product.metadata.localizedPriceString;
            }
        }
    }

    IEnumerator CountTo(int from, int to)
    {
        float animationDuration = 2.0f;
        int start = from;

        for (float timer = 0; timer < animationDuration; timer += Time.deltaTime)
        {
            float progress = timer / animationDuration;
            int numberToDisplay = (int)Mathf.Lerp(start, to, progress);

            TextCoinsCount.text = $"Coins: {numberToDisplay}";

            yield return null;
        }

        TextCoinsCount.text = $"Coins: {to}";

        yield return new WaitForEndOfFrame();
    }

    public void PurchaseProduct(string productId)
    {
        Product[] fetchedProducts = IStoreController.products.all;
        Product boughtProduct = fetchedProducts.FirstOrDefault((Product p) => Equals(p.definition.id, productId));
        if (boughtProduct != null)
        {
            Utils.Log($"Buying product with id: {productId}");
            IStoreController.InitiatePurchase(boughtProduct, $"quick devPayload=123,productId={productId}");
        }
        else
        {
            Utils.Log($"Not found product with id: {productId}");
        }
    }

    public void BtnBackClicked()
    {
        Utils.LoadScene(Constants.SceneName.MainMenu);
    }

    public void BtnInitNowggPaymentsClicked()
    {
        SetupNowggPayments();
    }
}
